#!/bin/bash
#
# Create new Split and update definitions for Boxed demo
#
# Parameters:
#   1 - Workspace Name
#   2 - Environment Name
#   3 - Traffic Type
#   4 - Split name
#   5 - Admin API Key (optional if SPLIT_API_KEY env var is set)
#
# Example:
#   CreateBoxSplit.sh Default Production user front_end_choose_boxes API_KEY
#
# Environment Variables:
#   SPLIT_API_KEY - Can be used instead of passing API key as argument
#

set -e

# --- Configuration ---
BASE_URL="https://api.split.io/internal/api/v2"
PAGE_LIMIT=20  # Number of items per page for paginated requests

# --- Parse Arguments ---
WORKSPACE_NAME="$1"
ENVIRONMENT_NAME="$2"
TRAFFIC_TYPE="$3"
SPLIT_NAME="$4"
API_KEY="${5:-$SPLIT_API_KEY}"

# --- Validation ---
if [ -z "$WORKSPACE_NAME" ] || [ -z "$ENVIRONMENT_NAME" ] || [ -z "$TRAFFIC_TYPE" ] || [ -z "$SPLIT_NAME" ]; then
  echo "Usage: $0 <workspace_name> <environment_name> <traffic_type> <split_name> [api_key]"
  echo ""
  echo "Arguments:"
  echo "  workspace_name   - Name of the Split workspace/Harness Project"
  echo "  environment_name - Environment name (e.g., Production)"
  echo "  traffic_type     - Traffic type (e.g., user)"
  echo "  split_name       - Name for the new Split/Feature Flag"
  echo "  api_key          - API key (optional if SPLIT_API_KEY env var is set)"
  exit 1
fi

if [ -z "$API_KEY" ]; then
  echo "Error: API key is required. Provide as 5th argument or set SPLIT_API_KEY environment variable."
  exit 1
fi

# --- Helper Functions ---

# Make an API request with fallback authentication
# Usage: api_request <method> <endpoint> [data]
api_request() {
  local method="$1"
  local endpoint="$2"
  local data="$3"
  local url="${BASE_URL}${endpoint}"
  local response
  local http_code

  # Build curl arguments
  local curl_args=(-s -w "\n%{http_code}" -X "$method" -H "Content-Type: application/json")
  
  if [ -n "$data" ]; then
    curl_args+=(-d "$data")
  fi

  # Try with x-api-key header first
  response=$(curl "${curl_args[@]}" -H "x-api-key: $API_KEY" "$url")
  http_code=$(echo "$response" | tail -n1)
  local body=$(echo "$response" | sed '$d')

  # If unauthorized (401/403), try with Bearer token
  if [ "$http_code" = "401" ] || [ "$http_code" = "403" ]; then
    echo "x-api-key auth failed, trying Bearer token..." >&2
    response=$(curl "${curl_args[@]}" -H "Authorization: Bearer $API_KEY" "$url")
    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')
  fi

  # Check for errors
  if [ "$http_code" -ge 400 ] 2>/dev/null; then
    echo "Error: API request failed with status $http_code" >&2
    echo "Response: $body" >&2
    return 1
  fi

  echo "$body"
}

# Find workspace ID by name with pagination support
# Usage: find_workspace_id <workspace_name>
# Returns: workspace ID or empty string if not found
find_workspace_id() {
  local workspace_name="$1"
  local offset=0
  local workspace_id=""
  local all_workspaces=""
  
  while true; do
    local response
    response=$(api_request GET "/workspaces?offset=$offset&limit=$PAGE_LIMIT")
    if [ $? -ne 0 ]; then
      echo "Failed to fetch workspaces (offset=$offset)" >&2
      return 1
    fi
    
    # Try to find the workspace in this page
    workspace_id=$(echo "$response" | jq -r ".objects[] | select(.name==\"$workspace_name\").id")
    
    if [ -n "$workspace_id" ] && [ "$workspace_id" != "null" ]; then
      # Found it!
      echo "$workspace_id"
      return 0
    fi
    
    # Collect workspace names for error reporting
    local page_names
    page_names=$(echo "$response" | jq -r '.objects[].name' 2>/dev/null)
    if [ -n "$page_names" ]; then
      if [ -n "$all_workspaces" ]; then
        all_workspaces="$all_workspaces"$'\n'"$page_names"
      else
        all_workspaces="$page_names"
      fi
    fi
    
    # Check if there are more pages
    local count
    count=$(echo "$response" | jq -r '.objects | length')
    
    if [ "$count" -lt "$PAGE_LIMIT" ] 2>/dev/null; then
      # No more pages, workspace not found
      echo "Error: Could not find workspace '$workspace_name'" >&2
      if [ -n "$all_workspaces" ]; then
        echo "Available workspaces:" >&2
        echo "$all_workspaces" >&2
      fi
      return 1
    fi
    
    # Move to next page
    offset=$((offset + PAGE_LIMIT))
    echo "Checking next page (offset=$offset)..." >&2
  done
}

# --- Main Script ---

echo "Fetching workspace ID for '$WORKSPACE_NAME'..."

# Get workspace ID with pagination support
workspaceId=$(find_workspace_id "$WORKSPACE_NAME")
if [ $? -ne 0 ] || [ -z "$workspaceId" ]; then
  exit 1
fi

echo "Workspace ID: $workspaceId"

# Create the split
echo "Creating split '$SPLIT_NAME'..."

split_data=$(cat <<EOF
{
  "name": "$SPLIT_NAME",
  "description": "Split created by CreateBoxSplit script"
}
EOF
)

create_response=$(api_request POST "/splits/ws/$workspaceId/trafficTypes/$TRAFFIC_TYPE" "$split_data")
if [ $? -ne 0 ]; then
  echo "Failed to create split"
  exit 1
fi

echo "Split created successfully"

# Configure the split with treatments
echo "Configuring split treatments in environment '$ENVIRONMENT_NAME'..."

treatment_data=$(cat <<'EOF'
{
  "treatments": [
    {
      "name": "premium",
      "description": "",
      "configurations": "{\"font_size\":\"large\",\"popup_message\":\"BigDiscount\",\"popup_value\":\"25%\"}"
    },
    {
      "name": "standard",
      "description": "",
      "configurations": "{\"font_size\":\"large\",\"popup_message\":\"Discount\",\"popup_value\":\"10%\"}"
    },
    {
      "name": "current",
      "description": "",
      "configurations": "{\"font_size\":\"large\",\"popup_message\":\"Thanksforboxing\",\"popup_value\":\"\"}"
    }
  ],
  "defaultTreatment": "current",
  "rules": [],
  "defaultRule": [
    {
      "treatment": "current",
      "size": 100
    }
  ]
}
EOF
)

config_response=$(api_request POST "/splits/ws/$workspaceId/$SPLIT_NAME/environments/$ENVIRONMENT_NAME" "$treatment_data")
if [ $? -ne 0 ]; then
  echo "Failed to configure split treatments"
  exit 1
fi

echo "Split '$SPLIT_NAME' created and configured successfully!"
echo "  Workspace: $WORKSPACE_NAME"
echo "  Environment: $ENVIRONMENT_NAME"
echo "  Traffic Type: $TRAFFIC_TYPE"
